#!/bin/bash

# ============================================
# ESPORTS TOURNAMENT PLATFORM - INSTALLATION SCRIPT
# ============================================

echo "╔════════════════════════════════════════╗"
echo "║  ESPORTS TOURNAMENT PLATFORM SETUP     ║"
echo "║  Free Fire & BGMI Tournament Website   ║"
echo "╚════════════════════════════════════════╝"
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${YELLOW}Warning: Not running as root. Some operations may fail.${NC}"
fi

echo -e "${BLUE}Step 1: Checking Requirements...${NC}"

# Check PHP
if command -v php &> /dev/null; then
    PHP_VERSION=$(php -v | head -n 1 | cut -d " " -f 2 | cut -f1-2 -d".")
    echo -e "${GREEN}✓${NC} PHP $PHP_VERSION found"
else
    echo -e "${RED}✗${NC} PHP not found. Please install PHP 8.0 or higher"
    exit 1
fi

# Check MySQL
if command -v mysql &> /dev/null; then
    echo -e "${GREEN}✓${NC} MySQL found"
else
    echo -e "${RED}✗${NC} MySQL not found. Please install MySQL 5.7 or higher"
    exit 1
fi

echo ""
echo -e "${BLUE}Step 2: Database Setup${NC}"
read -p "Enter MySQL username [root]: " DB_USER
DB_USER=${DB_USER:-root}

read -sp "Enter MySQL password: " DB_PASS
echo ""

read -p "Enter database name [esports_tournament_db]: " DB_NAME
DB_NAME=${DB_NAME:-esports_tournament_db}

# Create database
echo -e "${YELLOW}Creating database...${NC}"
mysql -u$DB_USER -p$DB_PASS -e "CREATE DATABASE IF NOT EXISTS $DB_NAME;" 2>/dev/null

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓${NC} Database created successfully"
else
    echo -e "${RED}✗${NC} Failed to create database"
    exit 1
fi

# Import schema
echo -e "${YELLOW}Importing schema...${NC}"
mysql -u$DB_USER -p$DB_PASS $DB_NAME < database/schema.sql 2>/dev/null

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓${NC} Schema imported successfully"
else
    echo -e "${RED}✗${NC} Failed to import schema"
    exit 1
fi

# Import seed data
read -p "Import sample test data? (y/n) [y]: " IMPORT_SEED
IMPORT_SEED=${IMPORT_SEED:-y}

if [ "$IMPORT_SEED" = "y" ]; then
    echo -e "${YELLOW}Importing test data...${NC}"
    mysql -u$DB_USER -p$DB_PASS $DB_NAME < database/seed.sql 2>/dev/null
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓${NC} Test data imported"
        echo -e "${YELLOW}Test Credentials:${NC}"
        echo "  Admin: admin / Admin@123"
        echo "  User: player1 / User@123"
    fi
fi

echo ""
echo -e "${BLUE}Step 3: Configuration${NC}"

# Update database config
echo -e "${YELLOW}Updating database configuration...${NC}"

cat > config/database.php << EOF
<?php
/**
 * Database Configuration
 */

return [
    'host' => 'localhost',
    'database' => '$DB_NAME',
    'username' => '$DB_USER',
    'password' => '$DB_PASS',
    'charset' => 'utf8mb4',
    'collation' => 'utf8mb4_unicode_ci',
    'options' => [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]
];
EOF

echo -e "${GREEN}✓${NC} Database configuration updated"

echo ""
echo -e "${BLUE}Step 4: File Permissions${NC}"

# Set permissions
echo -e "${YELLOW}Setting file permissions...${NC}"
chmod -R 755 . 2>/dev/null
chmod -R 644 . 2>/dev/null
find . -type d -exec chmod 755 {} \; 2>/dev/null
find . -type f -exec chmod 644 {} \; 2>/dev/null

echo -e "${GREEN}✓${NC} Permissions set"

echo ""
echo -e "${BLUE}Step 5: Creating Required Directories${NC}"

mkdir -p public/uploads
mkdir -p logs
chmod 777 public/uploads 2>/dev/null
chmod 777 logs 2>/dev/null

echo -e "${GREEN}✓${NC} Directories created"

echo ""
echo "╔════════════════════════════════════════╗"
echo "║       INSTALLATION COMPLETE! 🎉         ║"
echo "╚════════════════════════════════════════╝"
echo ""
echo -e "${GREEN}Your esports platform is ready!${NC}"
echo ""
echo "Next Steps:"
echo "1. Point your web server to the /public directory"
echo "2. Visit your website"
echo "3. Login with test credentials"
echo "4. Change admin password immediately"
echo "5. Configure payment gateway in config/app.php"
echo ""
echo "Important Security:"
echo "- Change all default passwords"
echo "- Set up SSL certificate"
echo "- Review config/app.php settings"
echo ""
echo -e "${BLUE}Documentation:${NC}"
echo "- README.md - Complete setup guide"
echo "- DEPLOYMENT_GUIDE.md - Code examples"
echo "- FINAL_SUMMARY.md - Quick reference"
echo ""
echo -e "${GREEN}Happy Gaming! 🎮${NC}"
